/*
  To use this code, connect the following 5 wires:
 Arduino : Si470x board
 3.3V : VCC
 GND : GND
 A5 : SCLK
 A4 : SDIO
 D2 : RST
 A0 : Trimpot (optional)
 */

  #include <Wire.h>
  #include <Bounce.h>
  #include <EEPROM.h>
  #include <LiquidCrystal.h>   
  #include "Adafruit_BMP085.h"

  LiquidCrystal lcd(13, 12, 11, 10, 9, 7); 

  Adafruit_BMP085 bmp; 

  int resetPin = 2;
  int SDIO = A4; //SDA/A4 on Arduino
  int SCLK = A5; //SCL/A5 on Arduino
  uint16_t si4703_registers[16]; //There are 16 registers, each 16 bits large

  #define FAIL  0
  #define SUCCESS  1

  #define SI4703 0x10 //0b._001.0000 = I2C address of Si4703 - note that the Wire function assumes non-left-shifted I2C address, not 0b.0010.000W
  #define I2C_FAIL_MAX  10 //This is the number of attempts we will try to contact the device before erroring out

  #define IN_EUROPE //Use this define to setup European FM reception. I wuz there for a day during testing (TEI 2011).
  #define SEEK_DOWN  0 //Direction used for seeking. Default is down
  #define SEEK_UP  1
  #define SKMODE  10 
  #define SEEKUP  9
  #define SEEK  8

//Define the register names
  #define POWERCFG  0x02
  #define CHANNEL  0x03
  #define SYSCONFIG1  0x04
  #define SYSCONFIG2  0x05
  #define STATUSRSSI  0x0A
  #define READCHAN  0x0B

  //Register 0x03 - CHANNEL
  #define TUNE  15

  //Register 0x04 - SYSCONFIG1
  #define RDS  12
  #define DE  11

  //Register 0x05 - SYSCONFIG2
  #define SPACE1  5
  #define SPACE0  4

  //Register 0x0A - STATUSRSSI
  #define RDSR  15
  #define STC  14
  #define SFBL  13
  #define AFCRL  12
  #define RDSS  11
  #define STEREO  8


  int HMC6352Address = 0x42;
  // This is calculated in the setup() function
  int slaveAddress;
  byte headingData[2];
  int i, headingValue;

  int fmFreq[10] = {985, 893, 915, 937, 958, 967, 971, 974, 1004, 1011};
  char* fmName[10] = {"Radio 1 ", "Radio 2 ", "Radio 3 ", "Radio 4 ", "Mersey  ",
                      "City FM ", "Buzz    ", "Rock FM ", "Smooth  ", "Classic "};
                      
  byte currentChannel = 1;  
  boolean lastValue = LOW;    // Used for 'Bounce' function for switch debouncer
  
  boolean Preset = true;
  boolean lastPreset = Preset;
  
  byte Mode = 1;               // Normal, Set Volume, Set Sea Level.
  byte vol = 6;
  float seaLevel;
  
  boolean seaLevelSet = false;
  
  unsigned long tim;          // Timer for LCD Backlight.
  int timSec;
 
  Bounce channelUp = Bounce( 6,5 );
  Bounce channelDown = Bounce(5, 5); 
  Bounce modeSwitch = Bounce(4, 5);
  Bounce PresetSeekButton = Bounce(3, 5);

  void setup() {

    pinMode(6, INPUT);
    pinMode(5, INPUT);     
    pinMode(4, INPUT);
    pinMode(3, INPUT);       
    
    pinMode(8, OUTPUT);    // LCD Backlight Power 
    
    // Powering the LCD from a spare OUTPUT pin ensures that the Arduino is up and running
    // before the LCD tries to initialize itself.
    pinMode(A2, OUTPUT);   // LCD Power. 
    digitalWrite(A2, HIGH);
    delay(200);  

    digitalWrite(8, HIGH);  // Turn on the LCD Backlight.
    
    tim = millis();  // Begin count to turn off the Backlight.  
    timSec = EEPROM.read(1);  
    if ((timSec < 5) || (timSec > 21)) timSec = 5;   
    

   // set up the LCD's number of columns and rows: 
   lcd.begin(16, 2);
   lcd.print("Walk-Pal...");  


  
   si4703_init(); //Init the Si4703 - we need to toggle SDIO before Wire.begin takes over.
   
   vol = EEPROM.read(2);
   if (vol > 16) vol = 6;
        si4703_readRegisters();                  //Read the current register set
        si4703_registers[SYSCONFIG2] &= 0xFFF0;  //Clear volume bits
        si4703_registers[SYSCONFIG2] |= vol;     //Set new volume
        si4703_updateRegisters();                //Update 
        
   lcd.setCursor(9,1);      
   if (EEPROM.read(3) == 1) {  
      Preset = true;  
      currentChannel = EEPROM.read(0);
      if ((currentChannel < 0) || (currentChannel > 9)) currentChannel = 0;
      gotoChannel(fmFreq[currentChannel]); 
      lcd.print(fmName[currentChannel]);
   } else {
      Preset = false;
      int Channel = EEPROMReadInt(4); 
      if (Channel > 1075) Channel = 1075;
      gotoChannel(Channel); 
      lcd.print(readChannel()/ 10);
      lcd.print(".");
      lcd.print(readChannel() % 10);
      lcd.print("   ");    
      
   }
  
   slaveAddress = HMC6352Address >> 1;   // This results in 0x21 as the address to pass to TWI

   bmp.begin();
   Wire.begin();
   seaLevel = bmp.readPressure() + 56;  // 56 Pa = my height diff.
   delay(200);

}

void loop() {
  
  if ((digitalRead(6) == HIGH) || (digitalRead(5) == HIGH) ||
      (digitalRead(4) == HIGH) || (digitalRead(3) == HIGH))  {        // Turn on LCD Backlight and initialize timer
    digitalWrite(8, HIGH);
    tim = millis();
  } 

  if ((millis() - tim > timSec * 1000)) {          // If timer > mode-set limit, turn LCD Backlight off...
    if  (timSec != 21) {
      digitalWrite(8, LOW);
    }
    Mode = 1;                                    // ...and restore normal display.
    EEPROMSave();
  }
  
  PresetSeekButton.update();
  int value = PresetSeekButton.read();
  if (( value == HIGH) && (lastPreset == LOW)) {
   if (Preset == true) {
     Preset = false;
     gotoChannel(EEPROMReadInt(4));
    } else {
     Preset = true;
     gotoChannel(fmFreq[currentChannel]);
    }
   
   lastPreset = HIGH;
  } else {
    lastPreset = LOW;
  }  
  
   modeSwitch.update();                            // Read Mode switch and increment at each press
  value = modeSwitch.read();
  if (( value == HIGH) && (lastValue == LOW)) {
   Mode++;
   if (Mode > 4) Mode = 1;
   lastValue = HIGH;
  } else {
    lastValue = LOW;
  }
   
  channelUp.update ( );
  value = channelUp.read();
  if ( (value == HIGH) && (lastValue == LOW) ) {
    lastValue = HIGH;
    if (Mode == 1) {
      if (Preset == true) {
        currentChannel++;
        if (currentChannel > 9) currentChannel = 0;
        gotoChannel(fmFreq[currentChannel]);  
      } else {
        seek(SEEK_UP);
      }
     }
    if (Mode == 2) {
      if (vol < 15) {
        si4703_readRegisters();                    //Read the current register set
        if(vol < 15) vol++;                        //Limit max volume to 0x000F
        si4703_registers[SYSCONFIG2] &= 0xFFF0;    //Clear volume bits
        si4703_registers[SYSCONFIG2] |= vol;       //Set new volume
        si4703_updateRegisters();                  //Update
      }  
    }
    if (Mode == 3) {
      seaLevel = seaLevel + 1;
      seaLevelSet = true;
      if (seaLevel > 103500) seaLevel = 80000;
    }
     if (Mode == 4) {
      timSec++;
      if (timSec > 21) timSec = 5;
    }   
   } else {
    lastValue = LOW;
 }
 
 
   channelDown.update();
   value = channelDown.read();
   if ( (value == HIGH) && (lastValue == LOW) ) {
    lastValue = HIGH;
   if (Mode == 1) {
     if (Preset == true) {
        currentChannel--;
        if (currentChannel < 0) currentChannel = 9;
        gotoChannel(fmFreq[currentChannel]);  
     } else {
        seek(SEEK_DOWN);
     }
    }  
    if (Mode == 2) {
      if (vol > 0) {
        si4703_readRegisters();                  //Read the current register set
        if(vol > 0) vol--;                       //Limit min volume to 0
        si4703_registers[SYSCONFIG2] &= 0xFFF0;  //Clear volume bits
        si4703_registers[SYSCONFIG2] |= vol;     //Set new volume
        si4703_updateRegisters();                //Update
      }  
    }
    if (Mode == 3) {
      seaLevel = seaLevel - 1;
      seaLevelSet = true;
      if (seaLevel < 80000) seaLevel = 103000;
    }
    if (Mode == 4) {                           // Set backlight timer
      timSec--;
      if (timSec <5 ) timSec = 21;
    }  
   } else {
    lastValue = LOW;
  }
 
  Wire.beginTransmission(slaveAddress);
  Wire.write("A");              // The "Get Data" command
  Wire.endTransmission();
  delay(10);                   // The HMC6352 needs at least a 70us (microsecond) delay
  // after this command.  Using 10ms just makes it safe
  // Read the 2 heading bytes, MSB first
  // The resulting 16bit word is the compass heading in 10th's of a degree
  // For example: a heading of 1345 would be 134.5 degrees
  Wire.requestFrom(slaveAddress, 2);        // Request the 2 byte heading (MSB comes first)
  i = 0;
  while(Wire.available() && i < 2)
  { 
    headingData[i] = Wire.read();
    i++;
  }
  headingValue = headingData[0]*256 + headingData[1];  // Put the MSB and LSB together

    writeLCD();

     if ((Mode == 3) && ((digitalRead(6) == HIGH) || (digitalRead(5) == HIGH))) {
       delay(100);        // Increment seaLevel setting faster than other settings
     } else {
      delay(800);
     }
//   if (digitalRead(4) == HIGH) delay(500); // Extra delay when switching between setting Modes
}

// Only write to EEPROM after LCD Backlight timeout to avoid excessive writes to EEPROM.

void EEPROMSave() {
  if (Preset == false) {
    EEPROM.write(3, 0);
    EEPROMWriteInt(4, readChannel());
  } else {
    EEPROM.write(3, 1);
    EEPROM.write(0,currentChannel);
  }
  EEPROM.write(2, vol);
  EEPROM.write(1, timSec);
  EEPROMWriteInt(6, int(seaLevel));
}

 //This function will write a 2 byte integer to the eeprom at the specified address and address + 1
  void EEPROMWriteInt(int p_address, int p_value) {
   byte lowByte = ((p_value >> 0) & 0xFF);
   byte highByte = ((p_value >> 8) & 0xFF);

   EEPROM.write(p_address, lowByte);
   EEPROM.write(p_address + 1, highByte);
  }

  //This function will read a 2 byte integer from the eeprom at the specified address and address + 1
  unsigned int EEPROMReadInt(int p_address) {
    byte lowByte = EEPROM.read(p_address);
    byte highByte = EEPROM.read(p_address + 1);
    return ((lowByte << 0) & 0xFF) + ((highByte << 8) & 0xFF00);
}

void writeLCD() {
  if (Mode == 1) {
    float temp;
    lcd.setCursor(0,0);
    lcd.print(int(bmp.readTemperature()));
    lcd.write(0xdf);
    lcd.print("C ");
    lcd.setCursor(5, 0);
    for (i = 0; i< 3; i++) {
     temp = temp + bmp.readPressure();
     delay(50);
    } 
    temp = temp / 3;
    lcd.print(temp/100, 1);

    lcd.print("    ");
    lcd.setCursor(13, 0);
    lcd.print("     ");
    lcd.setCursor(13,0);
    lcd.print(headingValue/10); 
    lcd.write(0xdf);  
    lcd.setCursor(0,1);
    if (seaLevelSet == false) {
      lcd.print("E ");
    } 
    temp = 0;
    for (i = 0; i< 6; i++) {
     temp = temp + bmp.readAltitude(seaLevel);
     delay(50);
    } 
    temp = temp / 6;
    lcd.print(temp, 1);
      lcd.print("m      ");
    

    lcd.setCursor(9,1);
    if (Preset == true) {
      lcd.print(fmName[currentChannel]);
    } else {
      lcd.print(readChannel()/ 10);
      lcd.print(".");
      lcd.print(readChannel() % 10);
      lcd.print("   ");
    }
  }
  
  if (Mode == 2) {
    lcd.clear();
    lcd.print("Volume: ");
    lcd.print(vol); //Read the current volume level
  }
 
  if (Mode == 3) {
    lcd.clear();
    lcd.print("Sea level kPa: ");
    lcd.setCursor(3, 1);
    lcd.print(seaLevel, 0);
  } 
  
 if (Mode == 4) {
    lcd.clear();
    lcd.print("Mode-set Timer: ");
    lcd.setCursor(3, 1);
    if (timSec < 21) {
      lcd.print(timSec);
      lcd.print(" seconds");
    } else {
      lcd.print("LED ON");
    }
  } 
}



//To get the Si4703 inito 2-wire mode, SEN needs to be high and SDIO needs to be low after a reset
//The breakout board has SEN pulled high, but also has SDIO pulled high. Therefore, after a normal power up
//The Si4703 will be in an unknown state. RST must be controlled
void si4703_init(void) {
 // Serial.println("Initializing I2C and Si4703");
  
  pinMode(resetPin, OUTPUT);
  pinMode(SDIO, OUTPUT); //SDIO is connected to A4 for I2C
  digitalWrite(SDIO, LOW); //A low SDIO indicates a 2-wire interface
  digitalWrite(resetPin, LOW); //Put Si4703 into reset
  delay(1); //Some delays while we allow pins to settle
  digitalWrite(resetPin, HIGH); //Bring Si4703 out of reset with SDIO set to low and SEN pulled high with on-board resistor
  delay(1); //Allow Si4703 to come out of reset

  Wire.begin(); //Now that the unit is reset and I2C inteface mode, we need to begin I2C

  si4703_readRegisters(); //Read the current register set
  //si4703_registers[0x07] = 0xBC04; //Enable the oscillator, from AN230 page 9, rev 0.5 (DOES NOT WORK, wtf Silicon Labs datasheet?)
  si4703_registers[0x07] = 0x8100; //Enable the oscillator, from AN230 page 9, rev 0.61 (works)
  si4703_updateRegisters(); //Update

  delay(500); //Wait for clock to settle - from AN230 page 9

  si4703_readRegisters(); //Read the current register set
  si4703_registers[POWERCFG] = 0x4001; //Enable the IC
  //  si4703_registers[POWERCFG] |= (1<<SMUTE) | (1<<DMUTE); //Disable Mute, disable softmute
  si4703_registers[SYSCONFIG1] |= (1<<RDS); //Enable RDS

#ifdef IN_EUROPE
    si4703_registers[SYSCONFIG1] |= (1<<DE); //50kHz Europe setup
  si4703_registers[SYSCONFIG2] |= (1<<SPACE0); //100kHz channel spacing for Europe
#else
  si4703_registers[SYSCONFIG2] &= ~(1<<SPACE1 | 1<<SPACE0) ; //Force 200kHz channel spacing for USA
#endif

  si4703_registers[SYSCONFIG2] &= 0xFFF0; //Clear volume bits
  si4703_registers[SYSCONFIG2] |= 0x0001; //Set volume to lowest
  si4703_updateRegisters(); //Update

  delay(110); //Max powerup time, from datasheet page 13
}

//Read the entire register control set from 0x00 to 0x0F
void si4703_readRegisters(void){

  //Si4703 begins reading from register upper register of 0x0A and reads to 0x0F, then loops to 0x00.
  Wire.requestFrom(SI4703, 32); //We want to read the entire register set from 0x0A to 0x09 = 32 bytes.

  while(Wire.available() < 32) ; //Wait for 16 words/32 bytes to come back from slave I2C device
  //We may want some time-out error here

  //Remember, register 0x0A comes in first so we have to shuffle the array around a bit
  for(int x = 0x0A ; ; x++) { //Read in these 32 bytes
    if(x == 0x10) x = 0; //Loop back to zero
    si4703_registers[x] = Wire.read() << 8;
    si4703_registers[x] |= Wire.read();
    if(x == 0x09) break; //We're done!
  }
}

//Write the current 9 control registers (0x02 to 0x07) to the Si4703
//It's a little weird, you don't write an I2C addres
//The Si4703 assumes you are writing to 0x02 first, then increments
byte si4703_updateRegisters(void) {

  Wire.beginTransmission(SI4703);
  //A write command automatically begins with register 0x02 so no need to send a write-to address
  //First we send the 0x02 to 0x07 control registers
  //In general, we should not write to registers 0x08 and 0x09
  for(int regSpot = 0x02 ; regSpot < 0x08 ; regSpot++) {
    byte high_byte = si4703_registers[regSpot] >> 8;
    byte low_byte = si4703_registers[regSpot] & 0x00FF;

    Wire.write(high_byte); //Upper 8 bits
    Wire.write(low_byte); //Lower 8 bits
  }

  //End this transmission
  byte ack = Wire.endTransmission();
  if(ack != 0) { //We have a problem! 
 //   Serial.print("Write Fail:"); //No ACK!
//    Serial.println(ack, DEC); //I2C error: 0 = success, 1 = data too long, 2 = rx NACK on address, 3 = rx NACK on data, 4 = other error
    return(FAIL);
  }

  return(SUCCESS);
}

//Given a channel, tune to it
//Channel is in MHz, so 973 will tune to 97.3MHz
//Note: gotoChannel will go to illegal channels (ie, greater than 110MHz)
//It's left to the user to limit these if necessary
//Actually, during testing the Si4703 seems to be internally limiting it at 87.5. Neat.
void gotoChannel(int newChannel){
  //Freq(MHz) = 0.200(in USA) * Channel + 87.5MHz
  //97.3 = 0.2 * Chan + 87.5
  //9.8 / 0.2 = 49
  newChannel *= 10; //973 * 10 = 9730
  newChannel -= 8750; //9730 - 8750 = 980

#ifdef IN_EUROPE
    newChannel /= 10; //980 / 10 = 98
#else
  newChannel /= 20; //980 / 20 = 49
#endif

  //These steps come from AN230 page 20 rev 0.5
  si4703_readRegisters();
  si4703_registers[CHANNEL] &= 0xFE00; //Clear out the channel bits
  si4703_registers[CHANNEL] |= newChannel; //Mask in the new channel
  si4703_registers[CHANNEL] |= (1<<TUNE); //Set the TUNE bit to start
  si4703_updateRegisters();

  //delay(60); //Wait 60ms - you can use or skip this delay

  //Poll to see if STC is set
  while(1) {
    si4703_readRegisters();
    if( (si4703_registers[STATUSRSSI] & (1<<STC)) != 0) break; //Tuning complete!
  }

  si4703_readRegisters();
  si4703_registers[CHANNEL] &= ~(1<<TUNE); //Clear the tune after a tune has completed
  si4703_updateRegisters();

  //Wait for the si4703 to clear the STC as well
  while(1) {
    si4703_readRegisters();
    if( (si4703_registers[STATUSRSSI] & (1<<STC)) == 0) break; //Tuning complete!
  }
}

//Seeks out the next available station
//Returns the freq if it made it
//Returns zero if failed
byte seek(byte seekDirection){
  si4703_readRegisters();

  //Set seek mode wrap bit
  //si4703_registers[POWERCFG] |= (1<<SKMODE); //Allow wrap
  si4703_registers[POWERCFG] &= ~(1<<SKMODE); //Disallow wrap - if you disallow wrap, you may want to tune to 87.5 first

  if(seekDirection == SEEK_DOWN) si4703_registers[POWERCFG] &= ~(1<<SEEKUP); //Seek down is the default upon reset
  else si4703_registers[POWERCFG] |= 1<<SEEKUP; //Set the bit to seek up

  si4703_registers[POWERCFG] |= (1<<SEEK); //Start seek

  si4703_updateRegisters(); //Seeking will now start

  //Poll to see if STC is set
  while(1) {
    si4703_readRegisters();
    if((si4703_registers[STATUSRSSI] & (1<<STC)) != 0) break; //Tuning complete!
    //Serial.println(readChannel());
  }

  si4703_readRegisters();
  int valueSFBL = si4703_registers[STATUSRSSI] & (1<<SFBL); //Store the value of SFBL
  si4703_registers[POWERCFG] &= ~(1<<SEEK); //Clear the seek bit after seek has completed
  si4703_updateRegisters();

  //Wait for the si4703 to clear the STC as well
  while(1) {
    si4703_readRegisters();
    if( (si4703_registers[STATUSRSSI] & (1<<STC)) == 0) break; //Tuning complete!
  }

  if(valueSFBL) { //The bit was set indicating we hit a band limit or failed to find a station
    return(FAIL);
  }
  
  return(SUCCESS);
}

//Reads the current channel from READCHAN
//Returns a number like 973 for 97.3MHz
int readChannel(void) {
  si4703_readRegisters();
  int channel = si4703_registers[READCHAN] & 0x03FF; //Mask out everything but the lower 10 bits

#ifdef IN_EUROPE
  //Freq(MHz) = 0.100(in Europe) * Channel + 87.5MHz
  //X = 0.1 * Chan + 87.5
  channel *= 1; //98 * 1 = 98 - I know this line is silly, but it makes the code look uniform
#else
  //Freq(MHz) = 0.200(in USA) * Channel + 87.5MHz
  //X = 0.2 * Chan + 87.5
  channel *= 2; //49 * 2 = 98
#endif

  channel += 875; //98 + 875 = 973
  return(channel);
}


